'use strict';

/* --------------------------------------------------------------
 multi_select.js 2016-11-08
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2016 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * ## Multi Select Widget
 *
 * This module serves as a wrapper of SumoSelect, a jQuery plugin that provides enhanced select-box functionality.
 * Bind this widget to a parent container and mark each child select-box element with the `data-multi_select-instance`
 * attribute.
 *
 * After the initialization of the widget all the marked elements will be converted into SumoSelect instances.
 *
 * ### Options
 *
 * **Options Source | `data-multi_select-source` | String | Optional**
 *
 * Provide a URL that will be used to fetch the options of the select box. The widget will perform a GET request to
 * the provided destination and expects a JSON array with the options:
 *
 * [
 *   {
 *     "value": "1", 
 *     "text": "Option #1"
 *   },
 *   {
 *     "value": "2", 
 *     "text": "Option #2"
 *   }
 * ]
 *
 * You can also pass other configuration directly in the parent element which will be used for every child instance.
 *
 *
 * ### Methods
 *
 * **Reload Options [AJAX]**
 *
 * You can use this method to refresh the options from the already provided data-multi_select-source or by providing
 * a new URL which will also be set as the data-source of the element. If the multi select has no URL then it will just
 * sync its values with the select element.
 *
 * * ```js
 * $('#my-multi-select').multi_select('reload', 'http://shop.de/options/source/url');
 * ```
 *
 * **Refresh Options**
 *
 * Update the multi-select widget with the state of the original select element. This method is useful after performing
 * changes in the original element and need to display them in the multi-select widget.
 *
 * ```js
 * $('#my-multi-select').multi_select('refresh');
 * ```
 *
 * ### Events
 * ```javascript
 * // Triggered when the multi-select widget has performed a "reload" method (after the AJAX call).
 * $('#my-multi-select').on('reload', function(event) {});
 * ```
 *
 * ### Example
 *
 * ```html
 * <form data-gx-widget="multi_select">
 *   <select data-multi_select-instance data-multi_select-source="http://shop.de/options-source-url"></select>
 * </form>
 * ```
 *
 * {@link http://hemantnegi.github.io/jquery.sumoselect}
 *
 * @module Admin/Widgets/multi_select
 * @requires jQuery-SumoSelect
 */
gx.widgets.module('multi_select', [jse.source + '/vendor/sumoselect/sumoselect.min.css', jse.source + '/vendor/sumoselect/jquery.sumoselect.min.js'], function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // VARIABLES
    // ------------------------------------------------------------------------

    /**
     * Module Selector
     *
     * @type {jQuery}
     */

    var $this = $(this);

    /**
     * Default Options
     *
     * @type {Object}
     */
    var defaults = {
        placeholder: jse.core.lang.translate('SELECT', 'general'),
        selectAll: true,
        csvDispCount: 2,
        captionFormat: '{0} ' + jse.core.lang.translate('selected', 'admin_labels'),
        locale: ['OK', jse.core.lang.translate('CANCEL', 'general'), jse.core.lang.translate('SELECT_ALL', 'general')]
    };

    /**
     * Final Options
     *
     * @type {Object}
     */
    var options = $.extend(true, {}, defaults, data);

    /**
     * Module Instance
     *
     * @type {Object}
     */
    var module = {};

    // ------------------------------------------------------------------------
    // FUNCTIONS
    // ------------------------------------------------------------------------

    /**
     * Add the "multi_select" method to the jQuery prototype.
     */
    function _addPublicMethod() {
        if ($.fn.multi_select) {
            return;
        }

        $.fn.extend({
            multi_select: function multi_select(action) {
                for (var _len = arguments.length, args = Array(_len > 1 ? _len - 1 : 0), _key = 1; _key < _len; _key++) {
                    args[_key - 1] = arguments[_key];
                }

                if (!$(this).is('select')) {
                    throw new Error('Called the "multi_select" method on an invalid object (select box expected).');
                }

                $.each(this, function () {
                    switch (action) {
                        case 'reload':
                            _reload.apply(undefined, [$(this)].concat(args));
                            break;

                        case 'refresh':
                            _refresh(this);
                    }
                });
            }
        });
    }

    /**
     * Fill a select box with the provided options.
     *
     * @param {jQuery} $select The select box to be filled.
     * @param {Object} options Array with { value: "...", "text": "..." } entries.
     */
    function _fillSelect($select, options) {
        $select.empty();
        $.each(options, function (index, option) {
            $select.append(new Option(option.text, option.value));
        });
    }

    /**
     * Reload the options from the source (data property) or the provided URL,
     *
     * @param {string} url Optional, if provided it will be used as the source of the data and will also update the
     * data-source property of the element.
     */
    function _reload($select, url) {
        url = url || $select.data('source');

        if (!url) {
            throw new Error('Multi Select Reload: Neither URL nor data-source contain a URL value.');
        }

        $select.data('source', url);

        $.getJSON(url).done(function (response) {
            _fillSelect($select, response);
            $select[0].sumo.reload();
            $select.trigger('reload');
        }).fail(function (jqxhr, textStatus, errorThrown) {
            jse.core.debug.error('Multi Select AJAX Error: ', jqxhr, textStatus, errorThrown);
        });
    }

    /**
     * Refresh the multi select instance depending the state of the original select element.
     *
     * @param {Node} select The DOM element to be refreshed.
     */
    function _refresh(select) {
        if (select.sumo === undefined) {
            throw new Error('Multi Select Refresh: The provided select element is not an instance of SumoSelect.', select);
        }

        select.sumo.reload();

        // Update the caption by simulating a click in an ".opt" element.
        _overrideSelectAllCaption.apply($(select.parentNode).find('.opt')[0]);
    }

    /**
     * Override the multi select caption when all elements are selected.
     *
     * This callback will override the caption because SumoSelect does not provide a setting for this text.
     */
    function _overrideSelectAllCaption() {
        var $optWrapper = $(this).parents('.optWrapper');
        var allCheckboxesChecked = $optWrapper.find('.opt.selected').length === $optWrapper.find('.opt').length;
        var atLeastOneCheckboxChecked = $optWrapper.find('.opt.selected').length > 0;
        var $selectAllCheckbox = $optWrapper.find('.select-all');

        $selectAllCheckbox.removeClass('partial-select');

        if (allCheckboxesChecked) {
            $optWrapper.siblings('.CaptionCont').children('span').text(jse.core.lang.translate('all_selected', 'admin_labels'));
        } else if (atLeastOneCheckboxChecked) {
            $selectAllCheckbox.addClass('partial-select');
        }
    }

    // ------------------------------------------------------------------------
    // INITIALIZATION
    // ------------------------------------------------------------------------

    module.init = function (done) {
        // Add public module method.
        _addPublicMethod();

        // Initialize the elements.
        $this.find('[data-multi_select-instance]').each(function () {
            var $select = $(this);

            $select.removeAttr('data-multi_select-instance');

            // Instantiate the widget without an AJAX request.
            $select.SumoSelect(options);

            if ($select.data('multi_selectSource') !== undefined) {
                // Remove the data attribute and store the value internally with the 'source' key.
                $select.data('source', $select.data('multi_selectSource'));
                $select.removeAttr('data-multi_select-source');

                // Fetch the options with an AJAX request.
                $.getJSON($select.data('multi_selectSource')).done(function (response) {
                    _fillSelect($select, response);
                    $select[0].sumo.unload();
                    $select.SumoSelect(options);
                    $select.trigger('reload');
                }).fail(function (jqxhr, textStatus, errorThrown) {
                    jse.core.debug.error('Multi Select AJAX Error: ', jqxhr, textStatus, errorThrown);
                });
            }
        });

        done();
    };

    // When the user clicks on the "Select All" option update the text with a custom translations. This has to
    // be done manually because there is no option for this text in SumoSelect.
    $this.on('click', '.select-all, .opt', _overrideSelectAllCaption);

    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
